package com.cav.mserver;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;

/**
 * A Mumps session over console.
 * 
 * @author Uri Schor
 */
public abstract class MumpsConsoleSession implements MumpsSession {

	private static Logger logger = Logger.getLogger(MumpsConsoleSession.class
			.getPackage().getName());

	/** Mumps code to run in order to check whether the session is alive */
	private static final String VALIDITY_CHECK_MCODE = "W 1";

	/** A print stream of the console's standard input */
	protected PrintStream consoleStdin;

	/** The console's standard output */
	protected InputStream consoleStdout;

	/** Is this session valid, i.e. still connected to Mumps console */
	private Boolean valid;

	/** Should Hebrew text within M commands be translated to 7-bit OLDCODE? */
	private static Boolean convertHebrewToOldcode;

	/** Should we leave NULL (0) characters or filter them out? */
	private static boolean leaveNullChars = Config.getBoolean(
			"m.leave_null_characters").booleanValue();
	
	/**
	 * An input stream that wraps the Mumps' console standard output and is used
	 * to read from it until the prompt, exclusive.
	 * 
	 * @author Uri Schor
	 */
	protected class MumpsConsoleInputStream extends InputStream {

		/** The current index within the prompt */
		private int index = 0;

		/** The wrapped input stream */
		private InputStream is;

		/** The Mumps session */
		private MumpsConsoleSession session;

		/** The prompt as int array for performance */
		private final char[] prompt = {'~', '~', '~', '>', ' '};

		/** Did we receive prompt by now */
		private boolean receivedPrompt = false;

		/** The current char from the prompt to retansmit, in case we read a
		 * prefix of the prompt and it turned out to be something else */
		private int retransmit = -1;

		/** The character we should retransmnit in case we read a prefix of the
		 * prompt and then it arrived and broke the prompt sequence */
		private int retransnitChar;

		/**
		 * Create a new input stream wrapper
		 * @param session The Mumps session
		 * @param in The original input stream (stdout of Mumps console)
		 */
		public MumpsConsoleInputStream(MumpsConsoleSession session,
				InputStream is) {
			this.is = is;
			this.session = session;
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#available()
		 */
		public int available() throws IOException {
			if (receivedPrompt) {
				return 0;
			}
			else {
				return is.available();
			}
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#close()
		 */
		public void close() throws IOException {
			// We do not close the actual stream, since it's reused
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		public boolean equals(Object obj) {
			return is.equals(obj);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		public int hashCode() {
			return is.hashCode();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#mark(int)
		 */
		public void mark(int readlimit) {
			is.mark(readlimit);
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#markSupported()
		 */
		public boolean markSupported() {
			return is.markSupported();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#read()
		 */
		public int read() throws IOException {
			try {
				// Check if the prompt was fully read
				if (receivedPrompt) {
					return -1;
				}

				// Check if we read a prefix of the promprt we should re-transmit
				if (retransmit > -1) {
					if (retransmit < index) {
						return prompt[retransmit++];
					}
					else if (retransmit == index) {
						index = 0;
						retransmit = -1;
						return retransnitChar;
					}
				}

				// Read the next character
				int c;
				while (true) {
					c = is.read();
					if (c == 0 && !leaveNullChars) {
						// ignore NULL
						continue;
					}
					if (c != prompt[index]) {
						// stop is the character is not the next in the prompt
						break;
					}
					++index;
					if (index == prompt.length) {
						// Prompt fully received
						receivedPrompt = true;
						// mark session as valid 
						session.validate(true);
						return -1;
					}
				}
				if (index > 0) {
					// We had read a prefix of the prompt, but it turns out it's
					// not the prompt after all.
					if (c != '~') {
						// patch the case of trailing tilde characters at the 
						// end of the output
						retransmit = 1;
						retransnitChar = c;
					}
					return prompt[0];
				}
				else {
					return c;
				}
			}
			catch (IOException e) {
				session.validate(false);
				throw e;
			}
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#reset()
		 */
		public void reset() throws IOException {
			is.reset();
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#skip(long)
		 */
		public long skip(long n) throws IOException {
			return is.skip(n);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			return is.toString();
		}

	}

	/* (non-Javadoc)
	 * @see com.cav.mserver.MumpsSession#close()
	 */
	abstract public void close();

	/* (non-Javadoc)
	 * @see com.cav.mserver.MumpsSession#execute(java.lang.String, java.util.Properties)
	 */
	public InputStream execute(String command, Map parameters) {
		StringBuffer commandBuf = new StringBuffer();

		// Iterate over parameters
		Iterator paramIter = parameters.entrySet().iterator();
		while (paramIter.hasNext()) {
			Map.Entry param = (Map.Entry)paramIter.next();
			if (commandBuf.length() > 0) {
				commandBuf.append(' ');
			}
			String value = null;
			if (param.getValue() instanceof String) {
				value = (String)param.getValue();
			}
			else if (param.getValue() instanceof String[]) {
				value = ((String[])param.getValue())[0];
			}
			else {
				throw new IllegalArgumentException(
						"Prameter should be of class String or String[]");
			}
			commandBuf.append("S ").append(param.getKey()).append('=').append(
				value);
		}
		if (commandBuf.length() > 0) {
			commandBuf.append(' ');
		}
		commandBuf.append(command);
		String commandStr = commandBuf.toString();
		commandStr = commandStr.replaceAll("^\\r\\n", "\\$C(13)_\\$C(10)_\"");
		commandStr = commandStr.replaceAll("\\r\\n$", "\"_\\$C(13)_\\$C(10)");
		commandStr = commandStr.replaceAll("\\r\\n", "\"_\\$C(13)_\\$C(10)_\"");
		commandStr = commandStr.replaceAll("^\\n", "\\$C(10)_\"");
		commandStr = commandStr.replaceAll("\\n$", "\"_\\$C(10)");
		commandStr = commandStr.replaceAll("\\n", "\"_\\$C(10)_\"");
		
		if (!command.equals(VALIDITY_CHECK_MCODE)) {
			logger.info("Executing " + commandStr);
		}
		executeCommand(commandStr);
		return new MumpsConsoleInputStream(this, consoleStdout);
	}

	/**
	 * Execute a Mumps command, without waiting for its output.
	 * @param command The Mumps command
	 */
	protected void executeCommand(String command) {
		// Clear leftovers on console stdout
		try {
			if (consoleStdout.available() > 0) {
				logger.warning("Leftovers on socket");
			}
			while (consoleStdout.available() > 0) {
				//System.err.print((char)consoleStdout.read());
				consoleStdout.read();
			}
		}
		catch (IOException e) {
		}

		// Run command
		consoleStdin.println(unicodeToASCIIHebrew(command));
		consoleStdin.flush();
	}

	/* (non-Javadoc)
	 * @see com.cav.mserver.MumpsSession#isRunning()
	 */
	public boolean validate() {
		if (valid != null) {
			// reset validity and return it
			boolean temp = valid.booleanValue();
			valid = null;
			return temp;
		}
		try {
			InputStream is = execute(VALIDITY_CHECK_MCODE, new HashMap());
			if (is.read() == -1) {
				// We sometimes get this on closed sessions
				return false;
			}
			while (is.read() != -1);
			is.close();
		}
		catch (IOException e) {
			return false;
		}
		return true;
	}

	/**
	 * Mark this session as valid or as invalid.
	 * @param valid Whether this session is valid
	 */
	protected void validate(boolean valid) {
		this.valid = valid ? Boolean.TRUE : Boolean.FALSE;
	}

	/**
	 * Convert each Unicode Hebrew character in a string, to 7-bit OLDCODE
	 * (Alef character is located at 96). 
	 * @param text
	 * @return
	 */
	private String unicodeToASCIIHebrew(String text) {
		// On the first call we check the configuration whether we should
		// convert to OLDCODE Hebrew
		if (convertHebrewToOldcode == null) {
			convertHebrewToOldcode = Config.getBoolean("m.oldcode_hebrew");
		}
		if (convertHebrewToOldcode.booleanValue()) {
			// Convert
	        StringBuffer sbuf = new StringBuffer(text.length());
	        for (int i = 0; i < text.length(); i++) {
	            int c = text.charAt(i);
	            if (c > 1487 && c < 1521) {
	            	c -= 1392;
	            }
	            if (c > 223 &&  c < 256) {
	            	c -= 128;
	            }
	            sbuf.append((char)c);
	        }
	        return sbuf.toString();
        }
		return text;
	}
}