package com.cav.mserver;

import java.io.*;
import java.util.logging.*;

/**
 * A Mumps telnet session.
 * @author Uri Schor
 */
public class MumpsTelnetSession extends MumpsConsoleSession {

	private static Logger logger = Logger.getLogger(MumpsTelnetSession.class
			.getPackage().getName());

	/** The telnet client */
	private TelnetClient telnetClient;

	/**
	 * Create a new TELNET session. Gets connection parameters from configuration.
	 * @see Config
	 * @throws IOException in case it's thrown durin the connection & login.
	 */
	public MumpsTelnetSession() throws IOException {
		init();
	}

	/**
	 * Create a new TELNET session.
	 * @param host host address to connect to
	 * @param port host port to connect to
	 * @param user TELNET user name
	 * @param password user's password
	 * @param timeout client socket timeout (SO_TIMEOUT), in milliseconds
	 * @throws IOException in case it's thrown durin the connection & login.
	 */
	public MumpsTelnetSession(String host, int port, String user, String password,
			int timeout) throws IOException {
		init(host, port, user, password, timeout);
	}
	
	/* (non-Javadoc)
	 * @see com.cav.mserver.MumpsConsoleSession#close()
	 */
	public void close() {
		try {
			executeCommand("HALT");
		}
		catch (Exception e) {
			logger.log(Level.WARNING, "Error sending HALT command", e);
		}
		try {
			telnetClient.disconnect();
		}
		catch (Exception e) {
			logger.log(Level.SEVERE, "Error disconnecting telnet", e);
		}
	}

	/* (non-Javadoc)
	 * @see com.cav.mserver.MumpsSession#init()
	 */
	public void init() throws IOException {
		String host = Config.getString("telnet.host");
		String portStr = Config.getString("telnet.port");
		int port = 23; // telnet service port
		if (portStr != null) {
			try {
				port = Integer.parseInt(portStr);
			}
			catch (NumberFormatException e) {
			}
		}
		String user = Config.getString("telnet.user");
		String password = Config.getString("telnet.password");
		String timeoutStr = Config.getString("telnet.timeout");
		int timeout = 0;
		if (timeoutStr != null) {
			try {
				timeout = Integer.parseInt(timeoutStr);
			}
			catch (NumberFormatException e) {
			}
		}
		init(host, port, user, password, timeout);
	}

	/**
	 * Initialize the TELNET session
	 * @param host
	 * @param port
	 * @param user
	 * @param password
	 * @param timeout
	 * @throws IOException
	 */
	private void init(String host, int port, String user, String password,
			int timeout) throws IOException {
		telnetClient = new TelnetClient(host, port, timeout);
		telnetClient.login(user, password);
		this.consoleStdin = new PrintStream(telnetClient.getOutputStream());
		this.consoleStdout = telnetClient.getInputStream();
	}

}